<?php
$pageTitle = 'Vendas - Sistema de Vendas';
include 'templates/header.php';
?>

<div class="card">
    <div class="card-header">
        <h2 class="card-title">Nova Venda</h2>
        <button class="btn btn-primary" onclick="Modal.open('vendaModal')">Registrar Venda</button>
    </div>
    
    <div class="table-container">
        <table class="table">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Cliente</th>
                    <th>Data</th>
                    <th>Total</th>
                    <th>Vendedor</th>
                    <th>Ações</th>
                </tr>
            </thead>
            <tbody id="vendasTable">
                <tr>
                    <td colspan="6" class="text-center">Carregando...</td>
                </tr>
            </tbody>
        </table>
    </div>
</div>

<!-- Modal Nova Venda -->
<div id="vendaModal" class="modal">
    <div class="modal-content" style="max-width: 800px;">
        <div class="modal-header">
            <h3 class="modal-title">Nova Venda</h3>
            <span class="close">&times;</span>
        </div>
        
        <form id="vendaForm">
            <div class="form-group">
                <label for="clienteSelect" class="form-label">Cliente</label>
                <select id="clienteSelect" name="cliente_id" class="form-control">
                    <option value="">Selecione um cliente (opcional)</option>
                </select>
            </div>
            
            <div class="form-group">
                <label class="form-label">Produtos</label>
                <div class="d-flex gap-2 mb-2">
                    <select id="produtoSelect" class="form-control">
                        <option value="">Selecione um produto</option>
                    </select>
                    <input type="number" id="quantidadeInput" class="form-control" placeholder="Qtd" min="1" style="max-width: 100px;">
                    <button type="button" class="btn btn-success" onclick="adicionarItem()">Adicionar</button>
                </div>
                
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Produto</th>
                                <th>Preço Unit.</th>
                                <th>Quantidade</th>
                                <th>Subtotal</th>
                                <th>Ações</th>
                            </tr>
                        </thead>
                        <tbody id="itensVendaTable">
                            <tr>
                                <td colspan="5" class="text-center">Nenhum item adicionado</td>
                            </tr>
                        </tbody>
                    </table>
                </div>
                
                <div class="text-right mt-2">
                    <strong>Total: <span id="totalVenda">R$ 0,00</span></strong>
                </div>
            </div>
            
            <div class="d-flex justify-between gap-2 mt-3">
                <button type="button" class="btn btn-warning" onclick="Modal.close('vendaModal')">Cancelar</button>
                <button type="submit" class="btn btn-primary">
                    <span class="btn-text">Finalizar Venda</span>
                    <span class="loading d-none"></span>
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Modal Detalhes da Venda -->
<div id="detalhesVendaModal" class="modal">
    <div class="modal-content" style="max-width: 700px;">
        <div class="modal-header">
            <h3 class="modal-title">Detalhes da Venda</h3>
            <span class="close">&times;</span>
        </div>
        
        <div id="detalhesVendaContent">
            <!-- Conteúdo será preenchido dinamicamente -->
        </div>
    </div>
</div>

<script>
let vendas = [];
let produtos = [];
let clientes = [];
let itensVenda = [];
let totalVenda = 0;

document.addEventListener('DOMContentLoaded', function() {
    listarVendas();
    carregarProdutos();
    carregarClientes();
    
    // Form submit
    document.getElementById('vendaForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        await finalizarVenda();
    });
});

async function listarVendas() {
    const response = await Vendas.listar();
    if (response.success) {
        vendas = response.data;
        renderizarTabelaVendas(vendas);
    } else {
        Utils.showAlert(response.message, 'danger');
    }
}

async function carregarProdutos() {
    const response = await Produtos.listar();
    if (response.success) {
        produtos = response.data;
        const select = document.getElementById('produtoSelect');
        select.innerHTML = '<option value="">Selecione um produto</option>' +
            produtos.map(produto => `<option value="${produto.id}" data-preco="${produto.preco}" data-estoque="${produto.estoque}">${produto.nome} - ${Utils.formatCurrency(produto.preco)} (Estoque: ${produto.estoque})</option>`).join('');
    }
}

async function carregarClientes() {
    const response = await Clientes.listar();
    if (response.success) {
        clientes = response.data;
        const select = document.getElementById('clienteSelect');
        select.innerHTML = '<option value="">Selecione um cliente (opcional)</option>' +
            clientes.map(cliente => `<option value="${cliente.id}">${cliente.nome}</option>`).join('');
    }
}

function renderizarTabelaVendas(vendas) {
    const tbody = document.getElementById('vendasTable');
    
    if (vendas.length === 0) {
        tbody.innerHTML = '<tr><td colspan="6" class="text-center">Nenhuma venda encontrada</td></tr>';
        return;
    }
    
    tbody.innerHTML = vendas.map(venda => `
        <tr>
            <td>${venda.id}</td>
            <td>${venda.cliente_nome || 'Cliente não informado'}</td>
            <td>${Utils.formatDate(venda.data_venda)}</td>
            <td>${Utils.formatCurrency(venda.total)}</td>
            <td>${venda.usuario_nome}</td>
            <td>
                <button class="btn btn-primary btn-sm" onclick="verDetalhesVenda(${venda.id})">Detalhes</button>
            </td>
        </tr>
    `).join('');
}

function adicionarItem() {
    const produtoSelect = document.getElementById('produtoSelect');
    const quantidadeInput = document.getElementById('quantidadeInput');
    
    const produtoId = produtoSelect.value;
    const quantidade = parseInt(quantidadeInput.value);
    
    if (!produtoId || !quantidade || quantidade <= 0) {
        Utils.showAlert('Selecione um produto e informe a quantidade', 'warning');
        return;
    }
    
    const produtoOption = produtoSelect.selectedOptions[0];
    const preco = parseFloat(produtoOption.dataset.preco);
    const estoque = parseInt(produtoOption.dataset.estoque);
    const nomeProduto = produtoOption.textContent.split(' - ')[0];
    
    if (quantidade > estoque) {
        Utils.showAlert('Quantidade maior que o estoque disponível', 'danger');
        return;
    }
    
    // Verificar se o produto já foi adicionado
    const itemExistente = itensVenda.find(item => item.produto_id == produtoId);
    if (itemExistente) {
        itemExistente.quantidade += quantidade;
        if (itemExistente.quantidade > estoque) {
            Utils.showAlert('Quantidade total maior que o estoque disponível', 'danger');
            itemExistente.quantidade -= quantidade;
            return;
        }
    } else {
        itensVenda.push({
            produto_id: produtoId,
            nome: nomeProduto,
            preco_unitario: preco,
            quantidade: quantidade
        });
    }
    
    // Limpar campos
    produtoSelect.value = '';
    quantidadeInput.value = '';
    
    renderizarItensVenda();
    calcularTotal();
}

function removerItem(index) {
    itensVenda.splice(index, 1);
    renderizarItensVenda();
    calcularTotal();
}

function renderizarItensVenda() {
    const tbody = document.getElementById('itensVendaTable');
    
    if (itensVenda.length === 0) {
        tbody.innerHTML = '<tr><td colspan="5" class="text-center">Nenhum item adicionado</td></tr>';
        return;
    }
    
    tbody.innerHTML = itensVenda.map((item, index) => `
        <tr>
            <td>${item.nome}</td>
            <td>${Utils.formatCurrency(item.preco_unitario)}</td>
            <td>${item.quantidade}</td>
            <td>${Utils.formatCurrency(item.preco_unitario * item.quantidade)}</td>
            <td>
                <button class="btn btn-danger btn-sm" onclick="removerItem(${index})">Remover</button>
            </td>
        </tr>
    `).join('');
}

function calcularTotal() {
    totalVenda = itensVenda.reduce((total, item) => total + (item.preco_unitario * item.quantidade), 0);
    document.getElementById('totalVenda').textContent = Utils.formatCurrency(totalVenda);
}

async function finalizarVenda() {
    if (itensVenda.length === 0) {
        Utils.showAlert('Adicione pelo menos um item à venda', 'warning');
        return;
    }
    
    const btnText = document.querySelector('#vendaForm .btn-text');
    const loading = document.querySelector('#vendaForm .loading');
    const submitBtn = document.querySelector('#vendaForm button[type="submit"]');
    
    // Mostrar loading
    btnText.classList.add('d-none');
    loading.classList.remove('d-none');
    submitBtn.disabled = true;
    
    try {
        const clienteId = document.getElementById('clienteSelect').value || null;
        
        const response = await Vendas.criar(clienteId, itensVenda);
        
        if (response.success) {
            Modal.close('vendaModal');
            limparFormularioVenda();
            listarVendas();
        }
    } finally {
        // Esconder loading
        btnText.classList.remove('d-none');
        loading.classList.add('d-none');
        submitBtn.disabled = false;
    }
}

function limparFormularioVenda() {
    document.getElementById('vendaForm').reset();
    itensVenda = [];
    totalVenda = 0;
    renderizarItensVenda();
    calcularTotal();
}

async function verDetalhesVenda(id) {
    const response = await Vendas.obter(id);
    if (response.success) {
        const venda = response.data;
        
        const content = `
            <div class="mb-3">
                <h4>Venda #${venda.id}</h4>
                <p><strong>Cliente:</strong> ${venda.cliente_nome || 'Não informado'}</p>
                <p><strong>Data:</strong> ${Utils.formatDate(venda.data_venda)}</p>
                <p><strong>Vendedor:</strong> ${venda.usuario_nome}</p>
                <p><strong>Total:</strong> ${Utils.formatCurrency(venda.total)}</p>
            </div>
            
            <h5>Itens da Venda</h5>
            <div class="table-container">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Produto</th>
                            <th>Preço Unit.</th>
                            <th>Quantidade</th>
                            <th>Subtotal</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${venda.itens.map(item => `
                            <tr>
                                <td>${item.produto_nome}</td>
                                <td>${Utils.formatCurrency(item.preco_unitario)}</td>
                                <td>${item.quantidade}</td>
                                <td>${Utils.formatCurrency(item.preco_unitario * item.quantidade)}</td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            </div>
        `;
        
        document.getElementById('detalhesVendaContent').innerHTML = content;
        Modal.open('detalhesVendaModal');
    } else {
        Utils.showAlert(response.message, 'danger');
    }
}
</script>

<?php include 'templates/footer.php'; ?>

