<?php
$pageTitle = 'Produtos - Sistema de Vendas';
include 'templates/header.php';
?>

<div class="card">
    <div class="card-header">
        <h2 class="card-title">Gerenciar Produtos</h2>
        <button class="btn btn-primary" onclick="Modal.open('produtoModal')">Novo Produto</button>
    </div>
    
    <div class="form-row mb-3">
        <div class="form-col">
            <input type="text" id="searchInput" class="form-control" placeholder="Buscar produtos...">
        </div>
        <div class="form-col" style="flex: 0 0 auto;">
            <button class="btn btn-primary" onclick="buscarProdutos()">Buscar</button>
            <button class="btn btn-warning" onclick="listarProdutos()">Limpar</button>
        </div>
    </div>
    
    <div class="table-container">
        <table class="table">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Nome</th>
                    <th>Descrição</th>
                    <th>Preço</th>
                    <th>Estoque</th>
                    <th>Ações</th>
                </tr>
            </thead>
            <tbody id="produtosTable">
                <tr>
                    <td colspan="6" class="text-center">Carregando...</td>
                </tr>
            </tbody>
        </table>
    </div>
</div>

<!-- Modal Produto -->
<div id="produtoModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3 class="modal-title" id="produtoModalTitle">Novo Produto</h3>
            <span class="close">&times;</span>
        </div>
        
        <form id="produtoForm">
            <input type="hidden" id="produtoId" name="id">
            
            <div class="form-group">
                <label for="produtoNome" class="form-label">Nome *</label>
                <input type="text" id="produtoNome" name="nome" class="form-control" required>
            </div>
            
            <div class="form-group">
                <label for="produtoDescricao" class="form-label">Descrição</label>
                <textarea id="produtoDescricao" name="descricao" class="form-control" rows="3"></textarea>
            </div>
            
            <div class="form-row">
                <div class="form-col">
                    <label for="produtoPreco" class="form-label">Preço *</label>
                    <input type="number" id="produtoPreco" name="preco" class="form-control" step="0.01" min="0" required>
                </div>
                
                <div class="form-col">
                    <label for="produtoEstoque" class="form-label">Estoque *</label>
                    <input type="number" id="produtoEstoque" name="estoque" class="form-control" min="0" required>
                </div>
            </div>
            
            <div class="d-flex justify-between gap-2 mt-3">
                <button type="button" class="btn btn-warning" onclick="Modal.close('produtoModal')">Cancelar</button>
                <button type="submit" class="btn btn-primary">
                    <span class="btn-text">Salvar</span>
                    <span class="loading d-none"></span>
                </button>
            </div>
        </form>
    </div>
</div>

<script>
let produtos = [];

document.addEventListener('DOMContentLoaded', function() {
    listarProdutos();
    
    // Form submit
    document.getElementById('produtoForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        await salvarProduto();
    });
    
    // Search on enter
    document.getElementById('searchInput').addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            buscarProdutos();
        }
    });
});

async function listarProdutos() {
    const response = await Produtos.listar();
    if (response.success) {
        produtos = response.data;
        renderizarTabela(produtos);
    } else {
        Utils.showAlert(response.message, 'danger');
    }
}

async function buscarProdutos() {
    const termo = document.getElementById('searchInput').value.trim();
    if (termo) {
        const response = await Produtos.buscar(termo);
        if (response.success) {
            produtos = response.data;
            renderizarTabela(produtos);
        } else {
            Utils.showAlert(response.message, 'danger');
        }
    } else {
        listarProdutos();
    }
}

function renderizarTabela(produtos) {
    const tbody = document.getElementById('produtosTable');
    
    if (produtos.length === 0) {
        tbody.innerHTML = '<tr><td colspan="6" class="text-center">Nenhum produto encontrado</td></tr>';
        return;
    }
    
    tbody.innerHTML = produtos.map(produto => `
        <tr>
            <td>${produto.id}</td>
            <td>${produto.nome}</td>
            <td>${produto.descricao || '-'}</td>
            <td>${Utils.formatCurrency(produto.preco)}</td>
            <td><span style="color: ${produto.estoque <= 10 ? '#dc3545' : '#28a745'}">${produto.estoque}</span></td>
            <td>
                <button class="btn btn-warning btn-sm" onclick="editarProduto(${produto.id})">Editar</button>
                <button class="btn btn-danger btn-sm" onclick="deletarProduto(${produto.id})">Deletar</button>
            </td>
        </tr>
    `).join('');
}

function novoProduto() {
    document.getElementById('produtoModalTitle').textContent = 'Novo Produto';
    document.getElementById('produtoForm').reset();
    document.getElementById('produtoId').value = '';
    Modal.open('produtoModal');
}

async function editarProduto(id) {
    const response = await Produtos.obter(id);
    if (response.success) {
        const produto = response.data;
        
        document.getElementById('produtoModalTitle').textContent = 'Editar Produto';
        document.getElementById('produtoId').value = produto.id;
        document.getElementById('produtoNome').value = produto.nome;
        document.getElementById('produtoDescricao').value = produto.descricao || '';
        document.getElementById('produtoPreco').value = produto.preco;
        document.getElementById('produtoEstoque').value = produto.estoque;
        
        Modal.open('produtoModal');
    } else {
        Utils.showAlert(response.message, 'danger');
    }
}

async function salvarProduto() {
    const btnText = document.querySelector('#produtoForm .btn-text');
    const loading = document.querySelector('#produtoForm .loading');
    const submitBtn = document.querySelector('#produtoForm button[type="submit"]');
    
    // Mostrar loading
    btnText.classList.add('d-none');
    loading.classList.remove('d-none');
    submitBtn.disabled = true;
    
    try {
        const id = document.getElementById('produtoId').value;
        const dados = {
            nome: document.getElementById('produtoNome').value,
            descricao: document.getElementById('produtoDescricao').value,
            preco: parseFloat(document.getElementById('produtoPreco').value),
            estoque: parseInt(document.getElementById('produtoEstoque').value)
        };
        
        let response;
        if (id) {
            response = await Produtos.atualizar(id, dados);
        } else {
            response = await Produtos.criar(dados);
        }
        
        if (response.success) {
            Modal.close('produtoModal');
            listarProdutos();
        }
    } finally {
        // Esconder loading
        btnText.classList.remove('d-none');
        loading.classList.add('d-none');
        submitBtn.disabled = false;
    }
}

async function deletarProduto(id) {
    const response = await Produtos.deletar(id);
    if (response.success) {
        listarProdutos();
    }
}
</script>

<?php include 'templates/footer.php'; ?>

