<?php
require_once '../../config/database.php';
require_once '../../includes/functions.php';

class Vendas {
    private $conn;
    
    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }
    
    public function listar($limit = 50, $offset = 0) {
        try {
            $query = "SELECT v.*, c.nome as cliente_nome, u.nome as usuario_nome 
                     FROM vendas v 
                     LEFT JOIN clientes c ON v.cliente_id = c.id 
                     LEFT JOIN usuarios u ON v.usuario_id = u.id 
                     ORDER BY v.data_venda DESC 
                     LIMIT :limit OFFSET :offset";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
            
            $vendas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return ['success' => true, 'data' => $vendas];
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao listar vendas: ' . $e->getMessage()];
        }
    }
    
    public function obter($id) {
        try {
            $query = "SELECT v.*, c.nome as cliente_nome, c.email as cliente_email, 
                            c.telefone as cliente_telefone, u.nome as usuario_nome 
                     FROM vendas v 
                     LEFT JOIN clientes c ON v.cliente_id = c.id 
                     LEFT JOIN usuarios u ON v.usuario_id = u.id 
                     WHERE v.id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            if ($stmt->rowCount() > 0) {
                $venda = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Buscar itens da venda
                $query = "SELECT iv.*, p.nome as produto_nome 
                         FROM itens_venda iv 
                         JOIN produtos p ON iv.produto_id = p.id 
                         WHERE iv.venda_id = :venda_id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':venda_id', $id);
                $stmt->execute();
                
                $venda['itens'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                return ['success' => true, 'data' => $venda];
            } else {
                return ['success' => false, 'message' => 'Venda não encontrada'];
            }
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao obter venda: ' . $e->getMessage()];
        }
    }
    
    public function criar($cliente_id, $itens) {
        try {
            $this->conn->beginTransaction();
            
            $total = 0;
            
            // Verificar estoque e calcular total
            foreach ($itens as $item) {
                $query = "SELECT estoque, preco FROM produtos WHERE id = :id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':id', $item['produto_id']);
                $stmt->execute();
                
                if ($stmt->rowCount() == 0) {
                    throw new Exception('Produto não encontrado: ' . $item['produto_id']);
                }
                
                $produto = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($produto['estoque'] < $item['quantidade']) {
                    throw new Exception('Estoque insuficiente para o produto ID: ' . $item['produto_id']);
                }
                
                $total += $produto['preco'] * $item['quantidade'];
            }
            
            // Criar a venda
            $query = "INSERT INTO vendas (cliente_id, usuario_id, total) VALUES (:cliente_id, :usuario_id, :total)";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':cliente_id', $cliente_id);
            $stmt->bindParam(':usuario_id', $_SESSION['user_id']);
            $stmt->bindParam(':total', $total);
            
            if (!$stmt->execute()) {
                throw new Exception('Erro ao criar venda');
            }
            
            $venda_id = $this->conn->lastInsertId();
            
            // Inserir itens da venda e atualizar estoque
            foreach ($itens as $item) {
                // Obter preço atual do produto
                $query = "SELECT preco FROM produtos WHERE id = :id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':id', $item['produto_id']);
                $stmt->execute();
                $produto = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Inserir item da venda
                $query = "INSERT INTO itens_venda (venda_id, produto_id, quantidade, preco_unitario) 
                         VALUES (:venda_id, :produto_id, :quantidade, :preco_unitario)";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':venda_id', $venda_id);
                $stmt->bindParam(':produto_id', $item['produto_id']);
                $stmt->bindParam(':quantidade', $item['quantidade']);
                $stmt->bindParam(':preco_unitario', $produto['preco']);
                
                if (!$stmt->execute()) {
                    throw new Exception('Erro ao inserir item da venda');
                }
                
                // Atualizar estoque
                $query = "UPDATE produtos SET estoque = estoque - :quantidade WHERE id = :id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':quantidade', $item['quantidade']);
                $stmt->bindParam(':id', $item['produto_id']);
                
                if (!$stmt->execute()) {
                    throw new Exception('Erro ao atualizar estoque');
                }
            }
            
            $this->conn->commit();
            
            return ['success' => true, 'message' => 'Venda criada com sucesso', 'id' => $venda_id];
            
        } catch (Exception $e) {
            $this->conn->rollback();
            return ['success' => false, 'message' => $e->getMessage()];
        } catch (PDOException $e) {
            $this->conn->rollback();
            return ['success' => false, 'message' => 'Erro ao criar venda: ' . $e->getMessage()];
        }
    }
    
    public function buscarPorPeriodo($data_inicio, $data_fim) {
        try {
            $query = "SELECT v.*, c.nome as cliente_nome, u.nome as usuario_nome 
                     FROM vendas v 
                     LEFT JOIN clientes c ON v.cliente_id = c.id 
                     LEFT JOIN usuarios u ON v.usuario_id = u.id 
                     WHERE DATE(v.data_venda) BETWEEN :data_inicio AND :data_fim 
                     ORDER BY v.data_venda DESC";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':data_inicio', $data_inicio);
            $stmt->bindParam(':data_fim', $data_fim);
            $stmt->execute();
            
            $vendas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return ['success' => true, 'data' => $vendas];
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao buscar vendas: ' . $e->getMessage()];
        }
    }
}

// Processar requisições AJAX
if ($_SERVER['REQUEST_METHOD'] === 'POST' || $_SERVER['REQUEST_METHOD'] === 'GET') {
    requireLogin();
    
    $vendas = new Vendas();
    $action = $_REQUEST['action'] ?? '';
    
    switch ($action) {
        case 'listar':
            $limit = intval($_GET['limit'] ?? 50);
            $offset = intval($_GET['offset'] ?? 0);
            $result = $vendas->listar($limit, $offset);
            sendJsonResponse($result);
            break;
            
        case 'obter':
            $id = intval($_GET['id'] ?? 0);
            if ($id <= 0) {
                sendJsonResponse(['success' => false, 'message' => 'ID inválido'], 400);
            }
            $result = $vendas->obter($id);
            sendJsonResponse($result);
            break;
            
        case 'criar':
            $cliente_id = intval($_POST['cliente_id'] ?? 0);
            $itens = json_decode($_POST['itens'] ?? '[]', true);
            
            if (empty($itens)) {
                sendJsonResponse(['success' => false, 'message' => 'Nenhum item informado'], 400);
            }
            
            // Validar itens
            foreach ($itens as $item) {
                if (!isset($item['produto_id']) || !isset($item['quantidade']) || 
                    intval($item['produto_id']) <= 0 || intval($item['quantidade']) <= 0) {
                    sendJsonResponse(['success' => false, 'message' => 'Dados dos itens inválidos'], 400);
                }
            }
            
            $result = $vendas->criar($cliente_id, $itens);
            sendJsonResponse($result);
            break;
            
        case 'buscar_periodo':
            $data_inicio = sanitizeInput($_GET['data_inicio'] ?? '');
            $data_fim = sanitizeInput($_GET['data_fim'] ?? '');
            
            if (empty($data_inicio) || empty($data_fim)) {
                sendJsonResponse(['success' => false, 'message' => 'Datas são obrigatórias'], 400);
            }
            
            $result = $vendas->buscarPorPeriodo($data_inicio, $data_fim);
            sendJsonResponse($result);
            break;
            
        default:
            sendJsonResponse(['success' => false, 'message' => 'Ação inválida'], 400);
    }
}
?>

