<?php
require_once '../../config/database.php';
require_once '../../includes/functions.php';

class Relatorios {
    private $conn;
    
    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }
    
    public function vendas_por_periodo($data_inicio, $data_fim) {
        try {
            $query = "SELECT 
                        DATE(v.data_venda) as data,
                        COUNT(*) as total_vendas,
                        SUM(v.total) as total_faturamento
                     FROM vendas v 
                     WHERE DATE(v.data_venda) BETWEEN :data_inicio AND :data_fim 
                     GROUP BY DATE(v.data_venda) 
                     ORDER BY DATE(v.data_venda)";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':data_inicio', $data_inicio);
            $stmt->bindParam(':data_fim', $data_fim);
            $stmt->execute();
            
            $dados = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return ['success' => true, 'data' => $dados];
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao gerar relatório: ' . $e->getMessage()];
        }
    }
    
    public function produtos_mais_vendidos($data_inicio = null, $data_fim = null, $limit = 10) {
        try {
            $where_clause = '';
            if ($data_inicio && $data_fim) {
                $where_clause = 'WHERE DATE(v.data_venda) BETWEEN :data_inicio AND :data_fim';
            }
            
            $query = "SELECT 
                        p.id,
                        p.nome,
                        SUM(iv.quantidade) as total_vendido,
                        SUM(iv.quantidade * iv.preco_unitario) as total_faturamento
                     FROM itens_venda iv
                     JOIN produtos p ON iv.produto_id = p.id
                     JOIN vendas v ON iv.venda_id = v.id
                     $where_clause
                     GROUP BY p.id, p.nome
                     ORDER BY total_vendido DESC
                     LIMIT :limit";
            
            $stmt = $this->conn->prepare($query);
            
            if ($data_inicio && $data_fim) {
                $stmt->bindParam(':data_inicio', $data_inicio);
                $stmt->bindParam(':data_fim', $data_fim);
            }
            
            $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
            $stmt->execute();
            
            $dados = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return ['success' => true, 'data' => $dados];
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao gerar relatório: ' . $e->getMessage()];
        }
    }
    
    public function estoque_baixo($limite = 10) {
        try {
            $query = "SELECT id, nome, estoque, preco 
                     FROM produtos 
                     WHERE estoque <= :limite 
                     ORDER BY estoque ASC";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':limite', $limite, PDO::PARAM_INT);
            $stmt->execute();
            
            $dados = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return ['success' => true, 'data' => $dados];
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao gerar relatório: ' . $e->getMessage()];
        }
    }
    
    public function clientes_mais_ativos($data_inicio = null, $data_fim = null, $limit = 10) {
        try {
            $where_clause = '';
            if ($data_inicio && $data_fim) {
                $where_clause = 'WHERE DATE(v.data_venda) BETWEEN :data_inicio AND :data_fim';
            }
            
            $query = "SELECT 
                        c.id,
                        c.nome,
                        c.email,
                        COUNT(v.id) as total_compras,
                        SUM(v.total) as total_gasto
                     FROM clientes c
                     JOIN vendas v ON c.id = v.cliente_id
                     $where_clause
                     GROUP BY c.id, c.nome, c.email
                     ORDER BY total_compras DESC
                     LIMIT :limit";
            
            $stmt = $this->conn->prepare($query);
            
            if ($data_inicio && $data_fim) {
                $stmt->bindParam(':data_inicio', $data_inicio);
                $stmt->bindParam(':data_fim', $data_fim);
            }
            
            $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
            $stmt->execute();
            
            $dados = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return ['success' => true, 'data' => $dados];
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao gerar relatório: ' . $e->getMessage()];
        }
    }
    
    public function resumo_geral($data_inicio = null, $data_fim = null) {
        try {
            $where_clause = '';
            if ($data_inicio && $data_fim) {
                $where_clause = 'WHERE DATE(v.data_venda) BETWEEN :data_inicio AND :data_fim';
            }
            
            // Total de vendas e faturamento
            $query = "SELECT 
                        COUNT(*) as total_vendas,
                        SUM(total) as total_faturamento,
                        AVG(total) as ticket_medio
                     FROM vendas v
                     $where_clause";
            $stmt = $this->conn->prepare($query);
            
            if ($data_inicio && $data_fim) {
                $stmt->bindParam(':data_inicio', $data_inicio);
                $stmt->bindParam(':data_fim', $data_fim);
            }
            
            $stmt->execute();
            $vendas_info = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Total de produtos
            $query = "SELECT COUNT(*) as total_produtos FROM produtos";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $produtos_info = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Total de clientes
            $query = "SELECT COUNT(*) as total_clientes FROM clientes";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $clientes_info = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Produtos com estoque baixo
            $query = "SELECT COUNT(*) as produtos_estoque_baixo FROM produtos WHERE estoque <= 10";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $estoque_info = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $resumo = array_merge($vendas_info, $produtos_info, $clientes_info, $estoque_info);
            
            return ['success' => true, 'data' => $resumo];
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao gerar relatório: ' . $e->getMessage()];
        }
    }
}

// Processar requisições AJAX
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    requireLogin();
    
    $relatorios = new Relatorios();
    $action = $_GET['action'] ?? '';
    
    switch ($action) {
        case 'vendas_periodo':
            $data_inicio = sanitizeInput($_GET['data_inicio'] ?? '');
            $data_fim = sanitizeInput($_GET['data_fim'] ?? '');
            
            if (empty($data_inicio) || empty($data_fim)) {
                sendJsonResponse(['success' => false, 'message' => 'Datas são obrigatórias'], 400);
            }
            
            $result = $relatorios->vendas_por_periodo($data_inicio, $data_fim);
            sendJsonResponse($result);
            break;
            
        case 'produtos_mais_vendidos':
            $data_inicio = sanitizeInput($_GET['data_inicio'] ?? '');
            $data_fim = sanitizeInput($_GET['data_fim'] ?? '');
            $limit = intval($_GET['limit'] ?? 10);
            
            $result = $relatorios->produtos_mais_vendidos($data_inicio ?: null, $data_fim ?: null, $limit);
            sendJsonResponse($result);
            break;
            
        case 'estoque_baixo':
            $limite = intval($_GET['limite'] ?? 10);
            $result = $relatorios->estoque_baixo($limite);
            sendJsonResponse($result);
            break;
            
        case 'clientes_mais_ativos':
            $data_inicio = sanitizeInput($_GET['data_inicio'] ?? '');
            $data_fim = sanitizeInput($_GET['data_fim'] ?? '');
            $limit = intval($_GET['limit'] ?? 10);
            
            $result = $relatorios->clientes_mais_ativos($data_inicio ?: null, $data_fim ?: null, $limit);
            sendJsonResponse($result);
            break;
            
        case 'resumo_geral':
            $data_inicio = sanitizeInput($_GET['data_inicio'] ?? '');
            $data_fim = sanitizeInput($_GET['data_fim'] ?? '');
            
            $result = $relatorios->resumo_geral($data_inicio ?: null, $data_fim ?: null);
            sendJsonResponse($result);
            break;
            
        default:
            sendJsonResponse(['success' => false, 'message' => 'Ação inválida'], 400);
    }
}
?>

