<?php
require_once '../../config/database.php';
require_once '../../includes/functions.php';

class Produtos {
    private $conn;
    
    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }
    
    public function listar($limit = 50, $offset = 0) {
        try {
            $query = "SELECT * FROM produtos ORDER BY nome LIMIT :limit OFFSET :offset";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
            
            $produtos = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return ['success' => true, 'data' => $produtos];
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao listar produtos: ' . $e->getMessage()];
        }
    }
    
    public function buscar($termo) {
        try {
            $query = "SELECT * FROM produtos WHERE nome LIKE :termo OR descricao LIKE :termo ORDER BY nome";
            $stmt = $this->conn->prepare($query);
            $termo = '%' . $termo . '%';
            $stmt->bindParam(':termo', $termo);
            $stmt->execute();
            
            $produtos = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return ['success' => true, 'data' => $produtos];
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao buscar produtos: ' . $e->getMessage()];
        }
    }
    
    public function obter($id) {
        try {
            $query = "SELECT * FROM produtos WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            if ($stmt->rowCount() > 0) {
                $produto = $stmt->fetch(PDO::FETCH_ASSOC);
                return ['success' => true, 'data' => $produto];
            } else {
                return ['success' => false, 'message' => 'Produto não encontrado'];
            }
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao obter produto: ' . $e->getMessage()];
        }
    }
    
    public function criar($nome, $descricao, $preco, $estoque) {
        try {
            $query = "INSERT INTO produtos (nome, descricao, preco, estoque) VALUES (:nome, :descricao, :preco, :estoque)";
            $stmt = $this->conn->prepare($query);
            
            $stmt->bindParam(':nome', $nome);
            $stmt->bindParam(':descricao', $descricao);
            $stmt->bindParam(':preco', $preco);
            $stmt->bindParam(':estoque', $estoque);
            
            if ($stmt->execute()) {
                $id = $this->conn->lastInsertId();
                return ['success' => true, 'message' => 'Produto criado com sucesso', 'id' => $id];
            } else {
                return ['success' => false, 'message' => 'Erro ao criar produto'];
            }
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao criar produto: ' . $e->getMessage()];
        }
    }
    
    public function atualizar($id, $nome, $descricao, $preco, $estoque) {
        try {
            $query = "UPDATE produtos SET nome = :nome, descricao = :descricao, preco = :preco, estoque = :estoque WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            
            $stmt->bindParam(':id', $id);
            $stmt->bindParam(':nome', $nome);
            $stmt->bindParam(':descricao', $descricao);
            $stmt->bindParam(':preco', $preco);
            $stmt->bindParam(':estoque', $estoque);
            
            if ($stmt->execute()) {
                return ['success' => true, 'message' => 'Produto atualizado com sucesso'];
            } else {
                return ['success' => false, 'message' => 'Erro ao atualizar produto'];
            }
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao atualizar produto: ' . $e->getMessage()];
        }
    }
    
    public function deletar($id) {
        try {
            $query = "DELETE FROM produtos WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                return ['success' => true, 'message' => 'Produto deletado com sucesso'];
            } else {
                return ['success' => false, 'message' => 'Erro ao deletar produto'];
            }
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao deletar produto: ' . $e->getMessage()];
        }
    }
    
    public function atualizarEstoque($id, $quantidade) {
        try {
            $query = "UPDATE produtos SET estoque = estoque + :quantidade WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->bindParam(':quantidade', $quantidade);
            
            if ($stmt->execute()) {
                return ['success' => true, 'message' => 'Estoque atualizado com sucesso'];
            } else {
                return ['success' => false, 'message' => 'Erro ao atualizar estoque'];
            }
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao atualizar estoque: ' . $e->getMessage()];
        }
    }
}

// Processar requisições AJAX
if ($_SERVER['REQUEST_METHOD'] === 'POST' || $_SERVER['REQUEST_METHOD'] === 'GET') {
    requireLogin();
    
    $produtos = new Produtos();
    $action = $_REQUEST['action'] ?? '';
    
    switch ($action) {
        case 'listar':
            $limit = intval($_GET['limit'] ?? 50);
            $offset = intval($_GET['offset'] ?? 0);
            $result = $produtos->listar($limit, $offset);
            sendJsonResponse($result);
            break;
            
        case 'buscar':
            $termo = sanitizeInput($_GET['termo'] ?? '');
            $result = $produtos->buscar($termo);
            sendJsonResponse($result);
            break;
            
        case 'obter':
            $id = intval($_GET['id'] ?? 0);
            if ($id <= 0) {
                sendJsonResponse(['success' => false, 'message' => 'ID inválido'], 400);
            }
            $result = $produtos->obter($id);
            sendJsonResponse($result);
            break;
            
        case 'criar':
            $nome = sanitizeInput($_POST['nome'] ?? '');
            $descricao = sanitizeInput($_POST['descricao'] ?? '');
            $preco = floatval($_POST['preco'] ?? 0);
            $estoque = intval($_POST['estoque'] ?? 0);
            
            if (empty($nome) || $preco <= 0) {
                sendJsonResponse(['success' => false, 'message' => 'Nome e preço são obrigatórios'], 400);
            }
            
            $result = $produtos->criar($nome, $descricao, $preco, $estoque);
            sendJsonResponse($result);
            break;
            
        case 'atualizar':
            $id = intval($_POST['id'] ?? 0);
            $nome = sanitizeInput($_POST['nome'] ?? '');
            $descricao = sanitizeInput($_POST['descricao'] ?? '');
            $preco = floatval($_POST['preco'] ?? 0);
            $estoque = intval($_POST['estoque'] ?? 0);
            
            if ($id <= 0 || empty($nome) || $preco <= 0) {
                sendJsonResponse(['success' => false, 'message' => 'Dados inválidos'], 400);
            }
            
            $result = $produtos->atualizar($id, $nome, $descricao, $preco, $estoque);
            sendJsonResponse($result);
            break;
            
        case 'deletar':
            $id = intval($_POST['id'] ?? 0);
            if ($id <= 0) {
                sendJsonResponse(['success' => false, 'message' => 'ID inválido'], 400);
            }
            
            $result = $produtos->deletar($id);
            sendJsonResponse($result);
            break;
            
        case 'atualizar_estoque':
            $id = intval($_POST['id'] ?? 0);
            $quantidade = intval($_POST['quantidade'] ?? 0);
            
            if ($id <= 0) {
                sendJsonResponse(['success' => false, 'message' => 'ID inválido'], 400);
            }
            
            $result = $produtos->atualizarEstoque($id, $quantidade);
            sendJsonResponse($result);
            break;
            
        default:
            sendJsonResponse(['success' => false, 'message' => 'Ação inválida'], 400);
    }
}
?>

