<?php
require_once '../../config/database.php';
require_once '../../includes/functions.php';

class Clientes {
    private $conn;
    
    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }
    
    public function listar($limit = 50, $offset = 0) {
        try {
            $query = "SELECT * FROM clientes ORDER BY nome LIMIT :limit OFFSET :offset";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
            
            $clientes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return ['success' => true, 'data' => $clientes];
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao listar clientes: ' . $e->getMessage()];
        }
    }
    
    public function buscar($termo) {
        try {
            $query = "SELECT * FROM clientes WHERE nome LIKE :termo OR email LIKE :termo OR telefone LIKE :termo ORDER BY nome";
            $stmt = $this->conn->prepare($query);
            $termo = '%' . $termo . '%';
            $stmt->bindParam(':termo', $termo);
            $stmt->execute();
            
            $clientes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return ['success' => true, 'data' => $clientes];
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao buscar clientes: ' . $e->getMessage()];
        }
    }
    
    public function obter($id) {
        try {
            $query = "SELECT * FROM clientes WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            
            if ($stmt->rowCount() > 0) {
                $cliente = $stmt->fetch(PDO::FETCH_ASSOC);
                return ['success' => true, 'data' => $cliente];
            } else {
                return ['success' => false, 'message' => 'Cliente não encontrado'];
            }
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao obter cliente: ' . $e->getMessage()];
        }
    }
    
    public function criar($nome, $email, $telefone, $endereco) {
        try {
            // Verificar se o email já existe (se fornecido)
            if (!empty($email)) {
                $query = "SELECT id FROM clientes WHERE email = :email";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':email', $email);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    return ['success' => false, 'message' => 'Email já cadastrado'];
                }
            }
            
            $query = "INSERT INTO clientes (nome, email, telefone, endereco) VALUES (:nome, :email, :telefone, :endereco)";
            $stmt = $this->conn->prepare($query);
            
            $stmt->bindParam(':nome', $nome);
            $stmt->bindParam(':email', $email);
            $stmt->bindParam(':telefone', $telefone);
            $stmt->bindParam(':endereco', $endereco);
            
            if ($stmt->execute()) {
                $id = $this->conn->lastInsertId();
                return ['success' => true, 'message' => 'Cliente criado com sucesso', 'id' => $id];
            } else {
                return ['success' => false, 'message' => 'Erro ao criar cliente'];
            }
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao criar cliente: ' . $e->getMessage()];
        }
    }
    
    public function atualizar($id, $nome, $email, $telefone, $endereco) {
        try {
            // Verificar se o email já existe em outro cliente (se fornecido)
            if (!empty($email)) {
                $query = "SELECT id FROM clientes WHERE email = :email AND id != :id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':email', $email);
                $stmt->bindParam(':id', $id);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    return ['success' => false, 'message' => 'Email já cadastrado para outro cliente'];
                }
            }
            
            $query = "UPDATE clientes SET nome = :nome, email = :email, telefone = :telefone, endereco = :endereco WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            
            $stmt->bindParam(':id', $id);
            $stmt->bindParam(':nome', $nome);
            $stmt->bindParam(':email', $email);
            $stmt->bindParam(':telefone', $telefone);
            $stmt->bindParam(':endereco', $endereco);
            
            if ($stmt->execute()) {
                return ['success' => true, 'message' => 'Cliente atualizado com sucesso'];
            } else {
                return ['success' => false, 'message' => 'Erro ao atualizar cliente'];
            }
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao atualizar cliente: ' . $e->getMessage()];
        }
    }
    
    public function deletar($id) {
        try {
            $query = "DELETE FROM clientes WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                return ['success' => true, 'message' => 'Cliente deletado com sucesso'];
            } else {
                return ['success' => false, 'message' => 'Erro ao deletar cliente'];
            }
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro ao deletar cliente: ' . $e->getMessage()];
        }
    }
}

// Processar requisições AJAX
if ($_SERVER['REQUEST_METHOD'] === 'POST' || $_SERVER['REQUEST_METHOD'] === 'GET') {
    requireLogin();
    
    $clientes = new Clientes();
    $action = $_REQUEST['action'] ?? '';
    
    switch ($action) {
        case 'listar':
            $limit = intval($_GET['limit'] ?? 50);
            $offset = intval($_GET['offset'] ?? 0);
            $result = $clientes->listar($limit, $offset);
            sendJsonResponse($result);
            break;
            
        case 'buscar':
            $termo = sanitizeInput($_GET['termo'] ?? '');
            $result = $clientes->buscar($termo);
            sendJsonResponse($result);
            break;
            
        case 'obter':
            $id = intval($_GET['id'] ?? 0);
            if ($id <= 0) {
                sendJsonResponse(['success' => false, 'message' => 'ID inválido'], 400);
            }
            $result = $clientes->obter($id);
            sendJsonResponse($result);
            break;
            
        case 'criar':
            $nome = sanitizeInput($_POST['nome'] ?? '');
            $email = sanitizeInput($_POST['email'] ?? '');
            $telefone = sanitizeInput($_POST['telefone'] ?? '');
            $endereco = sanitizeInput($_POST['endereco'] ?? '');
            
            if (empty($nome)) {
                sendJsonResponse(['success' => false, 'message' => 'Nome é obrigatório'], 400);
            }
            
            if (!empty($email) && !validateEmail($email)) {
                sendJsonResponse(['success' => false, 'message' => 'Email inválido'], 400);
            }
            
            $result = $clientes->criar($nome, $email, $telefone, $endereco);
            sendJsonResponse($result);
            break;
            
        case 'atualizar':
            $id = intval($_POST['id'] ?? 0);
            $nome = sanitizeInput($_POST['nome'] ?? '');
            $email = sanitizeInput($_POST['email'] ?? '');
            $telefone = sanitizeInput($_POST['telefone'] ?? '');
            $endereco = sanitizeInput($_POST['endereco'] ?? '');
            
            if ($id <= 0 || empty($nome)) {
                sendJsonResponse(['success' => false, 'message' => 'ID e nome são obrigatórios'], 400);
            }
            
            if (!empty($email) && !validateEmail($email)) {
                sendJsonResponse(['success' => false, 'message' => 'Email inválido'], 400);
            }
            
            $result = $clientes->atualizar($id, $nome, $email, $telefone, $endereco);
            sendJsonResponse($result);
            break;
            
        case 'deletar':
            $id = intval($_POST['id'] ?? 0);
            if ($id <= 0) {
                sendJsonResponse(['success' => false, 'message' => 'ID inválido'], 400);
            }
            
            $result = $clientes->deletar($id);
            sendJsonResponse($result);
            break;
            
        default:
            sendJsonResponse(['success' => false, 'message' => 'Ação inválida'], 400);
    }
}
?>

