<?php
require_once '../../config/database.php';
require_once '../../includes/functions.php';

class Auth {
    private $conn;
    
    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
    }
    
    public function login($email, $password) {
        try {
            $query = "SELECT id, nome, email, senha, tipo FROM usuarios WHERE email = :email";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':email', $email);
            $stmt->execute();
            
            if ($stmt->rowCount() > 0) {
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (verifyPassword($password, $user['senha'])) {
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['user_name'] = $user['nome'];
                    $_SESSION['user_email'] = $user['email'];
                    $_SESSION['user_type'] = $user['tipo'];
                    
                    return ['success' => true, 'message' => 'Login realizado com sucesso'];
                } else {
                    return ['success' => false, 'message' => 'Senha incorreta'];
                }
            } else {
                return ['success' => false, 'message' => 'Usuário não encontrado'];
            }
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro no servidor: ' . $e->getMessage()];
        }
    }
    
    public function logout() {
        session_destroy();
        return ['success' => true, 'message' => 'Logout realizado com sucesso'];
    }
    
    public function register($nome, $email, $password, $tipo = 'vendedor') {
        try {
            // Verificar se o email já existe
            $query = "SELECT id FROM usuarios WHERE email = :email";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':email', $email);
            $stmt->execute();
            
            if ($stmt->rowCount() > 0) {
                return ['success' => false, 'message' => 'Email já cadastrado'];
            }
            
            // Inserir novo usuário
            $query = "INSERT INTO usuarios (nome, email, senha, tipo) VALUES (:nome, :email, :senha, :tipo)";
            $stmt = $this->conn->prepare($query);
            
            $hashedPassword = generateHash($password);
            
            $stmt->bindParam(':nome', $nome);
            $stmt->bindParam(':email', $email);
            $stmt->bindParam(':senha', $hashedPassword);
            $stmt->bindParam(':tipo', $tipo);
            
            if ($stmt->execute()) {
                return ['success' => true, 'message' => 'Usuário cadastrado com sucesso'];
            } else {
                return ['success' => false, 'message' => 'Erro ao cadastrar usuário'];
            }
        } catch (PDOException $e) {
            return ['success' => false, 'message' => 'Erro no servidor: ' . $e->getMessage()];
        }
    }
}

// Processar requisições AJAX
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $auth = new Auth();
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'login':
            $email = sanitizeInput($_POST['email'] ?? '');
            $password = $_POST['password'] ?? '';
            
            if (empty($email) || empty($password)) {
                sendJsonResponse(['success' => false, 'message' => 'Email e senha são obrigatórios'], 400);
            }
            
            $result = $auth->login($email, $password);
            sendJsonResponse($result);
            break;
            
        case 'logout':
            $result = $auth->logout();
            sendJsonResponse($result);
            break;
            
        case 'register':
            $nome = sanitizeInput($_POST['nome'] ?? '');
            $email = sanitizeInput($_POST['email'] ?? '');
            $password = $_POST['password'] ?? '';
            $tipo = sanitizeInput($_POST['tipo'] ?? 'vendedor');
            
            if (empty($nome) || empty($email) || empty($password)) {
                sendJsonResponse(['success' => false, 'message' => 'Todos os campos são obrigatórios'], 400);
            }
            
            if (!validateEmail($email)) {
                sendJsonResponse(['success' => false, 'message' => 'Email inválido'], 400);
            }
            
            $result = $auth->register($nome, $email, $password, $tipo);
            sendJsonResponse($result);
            break;
            
        default:
            sendJsonResponse(['success' => false, 'message' => 'Ação inválida'], 400);
    }
}
?>

