<?php
$pageTitle = 'Clientes - Sistema de Vendas';
include 'templates/header.php';
?>

<div class="card">
    <div class="card-header">
        <h2 class="card-title">Gerenciar Clientes</h2>
        <button class="btn btn-primary" onclick="Modal.open('clienteModal')">Novo Cliente</button>
    </div>
    
    <div class="form-row mb-3">
        <div class="form-col">
            <input type="text" id="searchInput" class="form-control" placeholder="Buscar clientes...">
        </div>
        <div class="form-col" style="flex: 0 0 auto;">
            <button class="btn btn-primary" onclick="buscarClientes()">Buscar</button>
            <button class="btn btn-warning" onclick="listarClientes()">Limpar</button>
        </div>
    </div>
    
    <div class="table-container">
        <table class="table">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Nome</th>
                    <th>Email</th>
                    <th>Telefone</th>
                    <th>Endereço</th>
                    <th>Ações</th>
                </tr>
            </thead>
            <tbody id="clientesTable">
                <tr>
                    <td colspan="6" class="text-center">Carregando...</td>
                </tr>
            </tbody>
        </table>
    </div>
</div>

<!-- Modal Cliente -->
<div id="clienteModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3 class="modal-title" id="clienteModalTitle">Novo Cliente</h3>
            <span class="close">&times;</span>
        </div>
        
        <form id="clienteForm">
            <input type="hidden" id="clienteId" name="id">
            
            <div class="form-group">
                <label for="clienteNome" class="form-label">Nome *</label>
                <input type="text" id="clienteNome" name="nome" class="form-control" required>
            </div>
            
            <div class="form-group">
                <label for="clienteEmail" class="form-label">Email</label>
                <input type="email" id="clienteEmail" name="email" class="form-control">
            </div>
            
            <div class="form-group">
                <label for="clienteTelefone" class="form-label">Telefone</label>
                <input type="tel" id="clienteTelefone" name="telefone" class="form-control">
            </div>
            
            <div class="form-group">
                <label for="clienteEndereco" class="form-label">Endereço</label>
                <textarea id="clienteEndereco" name="endereco" class="form-control" rows="3"></textarea>
            </div>
            
            <div class="d-flex justify-between gap-2 mt-3">
                <button type="button" class="btn btn-warning" onclick="Modal.close('clienteModal')">Cancelar</button>
                <button type="submit" class="btn btn-primary">
                    <span class="btn-text">Salvar</span>
                    <span class="loading d-none"></span>
                </button>
            </div>
        </form>
    </div>
</div>

<script>
let clientes = [];

document.addEventListener('DOMContentLoaded', function() {
    listarClientes();
    
    // Form submit
    document.getElementById('clienteForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        await salvarCliente();
    });
    
    // Search on enter
    document.getElementById('searchInput').addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            buscarClientes();
        }
    });
});

async function listarClientes() {
    const response = await Clientes.listar();
    if (response.success) {
        clientes = response.data;
        renderizarTabela(clientes);
    } else {
        Utils.showAlert(response.message, 'danger');
    }
}

async function buscarClientes() {
    const termo = document.getElementById('searchInput').value.trim();
    if (termo) {
        const response = await Clientes.buscar(termo);
        if (response.success) {
            clientes = response.data;
            renderizarTabela(clientes);
        } else {
            Utils.showAlert(response.message, 'danger');
        }
    } else {
        listarClientes();
    }
}

function renderizarTabela(clientes) {
    const tbody = document.getElementById('clientesTable');
    
    if (clientes.length === 0) {
        tbody.innerHTML = '<tr><td colspan="6" class="text-center">Nenhum cliente encontrado</td></tr>';
        return;
    }
    
    tbody.innerHTML = clientes.map(cliente => `
        <tr>
            <td>${cliente.id}</td>
            <td>${cliente.nome}</td>
            <td>${cliente.email || '-'}</td>
            <td>${cliente.telefone || '-'}</td>
            <td>${cliente.endereco || '-'}</td>
            <td>
                <button class="btn btn-warning btn-sm" onclick="editarCliente(${cliente.id})">Editar</button>
                <button class="btn btn-danger btn-sm" onclick="deletarCliente(${cliente.id})">Deletar</button>
            </td>
        </tr>
    `).join('');
}

function novoCliente() {
    document.getElementById('clienteModalTitle').textContent = 'Novo Cliente';
    document.getElementById('clienteForm').reset();
    document.getElementById('clienteId').value = '';
    Modal.open('clienteModal');
}

async function editarCliente(id) {
    const response = await Clientes.obter(id);
    if (response.success) {
        const cliente = response.data;
        
        document.getElementById('clienteModalTitle').textContent = 'Editar Cliente';
        document.getElementById('clienteId').value = cliente.id;
        document.getElementById('clienteNome').value = cliente.nome;
        document.getElementById('clienteEmail').value = cliente.email || '';
        document.getElementById('clienteTelefone').value = cliente.telefone || '';
        document.getElementById('clienteEndereco').value = cliente.endereco || '';
        
        Modal.open('clienteModal');
    } else {
        Utils.showAlert(response.message, 'danger');
    }
}

async function salvarCliente() {
    const btnText = document.querySelector('#clienteForm .btn-text');
    const loading = document.querySelector('#clienteForm .loading');
    const submitBtn = document.querySelector('#clienteForm button[type="submit"]');
    
    // Mostrar loading
    btnText.classList.add('d-none');
    loading.classList.remove('d-none');
    submitBtn.disabled = true;
    
    try {
        const id = document.getElementById('clienteId').value;
        const dados = {
            nome: document.getElementById('clienteNome').value,
            email: document.getElementById('clienteEmail').value,
            telefone: document.getElementById('clienteTelefone').value,
            endereco: document.getElementById('clienteEndereco').value
        };
        
        let response;
        if (id) {
            response = await Clientes.atualizar(id, dados);
        } else {
            response = await Clientes.criar(dados);
        }
        
        if (response.success) {
            Modal.close('clienteModal');
            listarClientes();
        }
    } finally {
        // Esconder loading
        btnText.classList.remove('d-none');
        loading.classList.add('d-none');
        submitBtn.disabled = false;
    }
}

async function deletarCliente(id) {
    const response = await Clientes.deletar(id);
    if (response.success) {
        listarClientes();
    }
}
</script>

<?php include 'templates/footer.php'; ?>

