// Funções utilitárias
const Utils = {
    // Fazer requisições AJAX
    ajax: async function(url, options = {}) {
        try {
            const response = await fetch(url, {
                method: options.method || 'GET',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                    ...options.headers
                },
                body: options.body
            });
            
            const data = await response.json();
            return data;
        } catch (error) {
            console.error('Erro na requisição:', error);
            return { success: false, message: 'Erro de conexão' };
        }
    },

    // Mostrar alertas
    showAlert: function(message, type = 'info') {
        const alertDiv = document.createElement('div');
        alertDiv.className = `alert alert-${type}`;
        alertDiv.textContent = message;
        
        const container = document.querySelector('.container');
        container.insertBefore(alertDiv, container.firstChild);
        
        setTimeout(() => {
            alertDiv.remove();
        }, 5000);
    },

    // Formatar moeda
    formatCurrency: function(value) {
        return new Intl.NumberFormat('pt-BR', {
            style: 'currency',
            currency: 'BRL'
        }).format(value);
    },

    // Formatar data
    formatDate: function(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString('pt-BR') + ' ' + date.toLocaleTimeString('pt-BR');
    },

    // Validar email
    validateEmail: function(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    },

    // Serializar formulário
    serializeForm: function(form) {
        const formData = new FormData(form);
        const params = new URLSearchParams();
        
        for (const [key, value] of formData.entries()) {
            params.append(key, value);
        }
        
        return params.toString();
    }
};

// Gerenciador de modais
const Modal = {
    open: function(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.style.display = 'block';
            document.body.style.overflow = 'hidden';
        }
    },

    close: function(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.style.display = 'none';
            document.body.style.overflow = 'auto';
        }
    },

    closeAll: function() {
        const modals = document.querySelectorAll('.modal');
        modals.forEach(modal => {
            modal.style.display = 'none';
        });
        document.body.style.overflow = 'auto';
    }
};

// Gerenciador de autenticação
const Auth = {
    login: async function(email, password) {
        const response = await Utils.ajax('modules/auth/auth.php', {
            method: 'POST',
            body: `action=login&email=${encodeURIComponent(email)}&password=${encodeURIComponent(password)}`
        });

        if (response.success) {
            window.location.href = 'dashboard.php';
        } else {
            Utils.showAlert(response.message, 'danger');
        }

        return response;
    },

    logout: async function() {
        const response = await Utils.ajax('modules/auth/auth.php', {
            method: 'POST',
            body: 'action=logout'
        });

        if (response.success) {
            window.location.href = 'login.php';
        }

        return response;
    },

    register: async function(nome, email, password, tipo = 'vendedor') {
        const response = await Utils.ajax('modules/auth/auth.php', {
            method: 'POST',
            body: `action=register&nome=${encodeURIComponent(nome)}&email=${encodeURIComponent(email)}&password=${encodeURIComponent(password)}&tipo=${encodeURIComponent(tipo)}`
        });

        if (response.success) {
            Utils.showAlert(response.message, 'success');
        } else {
            Utils.showAlert(response.message, 'danger');
        }

        return response;
    }
};

// Gerenciador de produtos
const Produtos = {
    listar: async function(limit = 50, offset = 0) {
        return await Utils.ajax(`modules/produtos/produtos.php?action=listar&limit=${limit}&offset=${offset}`);
    },

    buscar: async function(termo) {
        return await Utils.ajax(`modules/produtos/produtos.php?action=buscar&termo=${encodeURIComponent(termo)}`);
    },

    obter: async function(id) {
        return await Utils.ajax(`modules/produtos/produtos.php?action=obter&id=${id}`);
    },

    criar: async function(dados) {
        const response = await Utils.ajax('modules/produtos/produtos.php', {
            method: 'POST',
            body: `action=criar&nome=${encodeURIComponent(dados.nome)}&descricao=${encodeURIComponent(dados.descricao)}&preco=${dados.preco}&estoque=${dados.estoque}`
        });

        if (response.success) {
            Utils.showAlert(response.message, 'success');
        } else {
            Utils.showAlert(response.message, 'danger');
        }

        return response;
    },

    atualizar: async function(id, dados) {
        const response = await Utils.ajax('modules/produtos/produtos.php', {
            method: 'POST',
            body: `action=atualizar&id=${id}&nome=${encodeURIComponent(dados.nome)}&descricao=${encodeURIComponent(dados.descricao)}&preco=${dados.preco}&estoque=${dados.estoque}`
        });

        if (response.success) {
            Utils.showAlert(response.message, 'success');
        } else {
            Utils.showAlert(response.message, 'danger');
        }

        return response;
    },

    deletar: async function(id) {
        if (confirm('Tem certeza que deseja deletar este produto?')) {
            const response = await Utils.ajax('modules/produtos/produtos.php', {
                method: 'POST',
                body: `action=deletar&id=${id}`
            });

            if (response.success) {
                Utils.showAlert(response.message, 'success');
            } else {
                Utils.showAlert(response.message, 'danger');
            }

            return response;
        }
        return { success: false, message: 'Operação cancelada' };
    }
};

// Gerenciador de clientes
const Clientes = {
    listar: async function(limit = 50, offset = 0) {
        return await Utils.ajax(`modules/clientes/clientes.php?action=listar&limit=${limit}&offset=${offset}`);
    },

    buscar: async function(termo) {
        return await Utils.ajax(`modules/clientes/clientes.php?action=buscar&termo=${encodeURIComponent(termo)}`);
    },

    obter: async function(id) {
        return await Utils.ajax(`modules/clientes/clientes.php?action=obter&id=${id}`);
    },

    criar: async function(dados) {
        const response = await Utils.ajax('modules/clientes/clientes.php', {
            method: 'POST',
            body: `action=criar&nome=${encodeURIComponent(dados.nome)}&email=${encodeURIComponent(dados.email)}&telefone=${encodeURIComponent(dados.telefone)}&endereco=${encodeURIComponent(dados.endereco)}`
        });

        if (response.success) {
            Utils.showAlert(response.message, 'success');
        } else {
            Utils.showAlert(response.message, 'danger');
        }

        return response;
    },

    atualizar: async function(id, dados) {
        const response = await Utils.ajax('modules/clientes/clientes.php', {
            method: 'POST',
            body: `action=atualizar&id=${id}&nome=${encodeURIComponent(dados.nome)}&email=${encodeURIComponent(dados.email)}&telefone=${encodeURIComponent(dados.telefone)}&endereco=${encodeURIComponent(dados.endereco)}`
        });

        if (response.success) {
            Utils.showAlert(response.message, 'success');
        } else {
            Utils.showAlert(response.message, 'danger');
        }

        return response;
    },

    deletar: async function(id) {
        if (confirm('Tem certeza que deseja deletar este cliente?')) {
            const response = await Utils.ajax('modules/clientes/clientes.php', {
                method: 'POST',
                body: `action=deletar&id=${id}`
            });

            if (response.success) {
                Utils.showAlert(response.message, 'success');
            } else {
                Utils.showAlert(response.message, 'danger');
            }

            return response;
        }
        return { success: false, message: 'Operação cancelada' };
    }
};

// Gerenciador de vendas
const Vendas = {
    listar: async function(limit = 50, offset = 0) {
        return await Utils.ajax(`modules/vendas/vendas.php?action=listar&limit=${limit}&offset=${offset}`);
    },

    obter: async function(id) {
        return await Utils.ajax(`modules/vendas/vendas.php?action=obter&id=${id}`);
    },

    criar: async function(clienteId, itens) {
        const response = await Utils.ajax('modules/vendas/vendas.php', {
            method: 'POST',
            body: `action=criar&cliente_id=${clienteId}&itens=${encodeURIComponent(JSON.stringify(itens))}`
        });

        if (response.success) {
            Utils.showAlert(response.message, 'success');
        } else {
            Utils.showAlert(response.message, 'danger');
        }

        return response;
    }
};

// Gerenciador de relatórios
const Relatorios = {
    resumoGeral: async function(dataInicio = '', dataFim = '') {
        let url = 'modules/relatorios/relatorios.php?action=resumo_geral';
        if (dataInicio && dataFim) {
            url += `&data_inicio=${dataInicio}&data_fim=${dataFim}`;
        }
        return await Utils.ajax(url);
    },

    vendasPorPeriodo: async function(dataInicio, dataFim) {
        return await Utils.ajax(`modules/relatorios/relatorios.php?action=vendas_periodo&data_inicio=${dataInicio}&data_fim=${dataFim}`);
    },

    produtosMaisVendidos: async function(dataInicio = '', dataFim = '', limit = 10) {
        let url = `modules/relatorios/relatorios.php?action=produtos_mais_vendidos&limit=${limit}`;
        if (dataInicio && dataFim) {
            url += `&data_inicio=${dataInicio}&data_fim=${dataFim}`;
        }
        return await Utils.ajax(url);
    },

    estoqueBaixo: async function(limite = 10) {
        return await Utils.ajax(`modules/relatorios/relatorios.php?action=estoque_baixo&limite=${limite}`);
    },

    clientesMaisAtivos: async function(dataInicio = '', dataFim = '', limit = 10) {
        let url = `modules/relatorios/relatorios.php?action=clientes_mais_ativos&limit=${limit}`;
        if (dataInicio && dataFim) {
            url += `&data_inicio=${dataInicio}&data_fim=${dataFim}`;
        }
        return await Utils.ajax(url);
    }
};

// Event listeners globais
document.addEventListener('DOMContentLoaded', function() {
    // Fechar modais ao clicar no X ou fora do modal
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('close') || e.target.classList.contains('modal')) {
            Modal.closeAll();
        }
    });

    // Fechar modais com ESC
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            Modal.closeAll();
        }
    });

    // Logout
    const logoutBtn = document.querySelector('.logout-btn');
    if (logoutBtn) {
        logoutBtn.addEventListener('click', function(e) {
            e.preventDefault();
            Auth.logout();
        });
    }

    // Navegação ativa
    const currentPage = window.location.pathname.split('/').pop();
    const navLinks = document.querySelectorAll('.nav-item a');
    navLinks.forEach(link => {
        if (link.getAttribute('href') === currentPage) {
            link.classList.add('active');
        }
    });
});

// Exportar para uso global
window.Utils = Utils;
window.Modal = Modal;
window.Auth = Auth;
window.Produtos = Produtos;
window.Clientes = Clientes;
window.Vendas = Vendas;
window.Relatorios = Relatorios;

